<?php
require_once 'includes/config.php';
require_once 'includes/db.php';

// Validate project ID
if (!isset($_GET['id']) || !ctype_digit($_GET['id'])) {
    header('HTTP/1.1 400 Bad Request');
    header('Location: projects.php');
    exit();
}

$projectId = (int)$_GET['id'];

try {
    // Get project details
    $stmt = $pdo->prepare("SELECT * FROM projects WHERE id = ?");
    $stmt->execute([$projectId]);
    $project = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Redirect if project not found
    if (!$project) {
        header('HTTP/1.1 404 Not Found');
        header('Location: projects.php');
        exit();
    }
    
    // Process project data
    $project['client'] = $project['client'] ?? 'Not specified';
    $project['location'] = $project['location'] ?? 'Location not available';
    $project['status'] = (!empty($project['completed_date']) && strtotime($project['completed_date']) > time()) 
        ? 'ongoing' 
        : 'completed';
    
    // Set page title
    $pageTitle = htmlspecialchars($project['title']) . ' | Project Details';
    
    // Get related projects (excluding current one)
    $relatedProjects = [];
    $relatedStmt = $pdo->prepare("
        SELECT id, title, image_url, client, location, completed_date 
        FROM projects 
        WHERE id != ? 
        ORDER BY completed_date DESC 
        LIMIT 3
    ");
    $relatedStmt->execute([$projectId]);
    $relatedProjects = $relatedStmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Database error in project.php: " . $e->getMessage());
    header('HTTP/1.1 500 Internal Server Error');
    include('includes/header.php');
    echo '<div class="container"><div class="alert alert-danger">We encountered an error loading this project. Please try again later.</div></div>';
    include('includes/footer.php');
    exit();
}

include('includes/header.php');
?>

<!-- Main Content Section -->
<main class="project-detail-page">
    <div class="container">
        <!-- Breadcrumb Navigation -->
        <nav class="breadcrumb-nav">
            <ol>
                <li><a href="index.php">Home</a></li>
                <li><a href="projects.php">Projects</a></li>
                <li><?= htmlspecialchars($project['title']) ?></li>
            </ol>
        </nav>

        <!-- Project Header -->
        <section class="project-header">
            <div class="project-title-wrapper">
                <h1><?= htmlspecialchars($project['title']) ?></h1>
                <span class="project-status <?= $project['status'] ?>">
                    <?= ucfirst($project['status']) ?>
                </span>
            </div>
            
            <div class="project-meta">
                <?php if (!empty($project['client'])): ?>
                    <div class="meta-item">
                        <i class="fas fa-building"></i>
                        <span>Client: <?= htmlspecialchars($project['client']) ?></span>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($project['location'])): ?>
                    <div class="meta-item">
                        <i class="fas fa-map-marker-alt"></i>
                        <span>Location: <?= htmlspecialchars($project['location']) ?></span>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($project['completed_date'])): ?>
                    <div class="meta-item">
                        <i class="far fa-calendar-alt"></i>
                        <span>
                            <?= $project['status'] === 'ongoing' ? 'Expected Completion' : 'Completed' ?>: 
                            <?= date('F Y', strtotime($project['completed_date'])) ?>
                        </span>
                    </div>
                <?php endif; ?>
            </div>
        </section>

        <!-- Project Image -->
        <section class="project-main-image">
            <img src="<?= htmlspecialchars($project['image_url']) ?>" 
                 alt="<?= htmlspecialchars($project['title']) ?>" 
                 class="featured-image">
        </section>

        <!-- Project Description -->
        <section class="project-description">
            <h2>Project Overview</h2>
            <div class="description-content">
                <?= !empty($project['description']) 
                    ? nl2br(htmlspecialchars($project['description'])) 
                    : '<p>No detailed description available for this project.</p>' ?>
            </div>
        </section>

        <!-- Related Projects -->
        <?php if (!empty($relatedProjects)): ?>
        <section class="related-projects">
            <h2>Other Projects You Might Like</h2>
            <div class="related-projects-grid">
                <?php foreach ($relatedProjects as $related): ?>
                    <article class="related-project">
                        <a href="project.php?id=<?= $related['id'] ?>">
                            <img src="<?= htmlspecialchars($related['image_url']) ?>" 
                                 alt="<?= htmlspecialchars($related['title']) ?>">
                            <div class="related-project-info">
                                <h3><?= htmlspecialchars($related['title']) ?></h3>
                                <?php if (!empty($related['client'])): ?>
                                    <p class="related-client"><?= htmlspecialchars($related['client']) ?></p>
                                <?php endif; ?>
                            </div>
                        </a>
                    </article>
                <?php endforeach; ?>
            </div>
        </section>
        <?php endif; ?>

        <!-- Call to Action -->
        <section class="project-cta">
            <h3>Have a Similar Project in Mind?</h3>
            <p>Contact us today to discuss your construction needs</p>
            <div class="cta-buttons">
                <a href="contact.php" class="btn btn-primary">Get a Free Quote</a>
                <a href="projects.php" class="btn btn-outline">View All Projects</a>
            </div>
        </section>
    </div>
</main>

<?php include('includes/footer.php'); ?>

<style>
/* Project Detail Page Specific Styles */
.project-detail-page {
    padding: 2rem 0;
}

.breadcrumb-nav ol {
    display: flex;
    gap: 0.5rem;
    list-style: none;
    padding: 0;
    margin: 0 0 2rem;
    font-size: 0.9rem;
}

.breadcrumb-nav a {
    color: var(--primary-color);
    text-decoration: none;
}

.breadcrumb-nav li:not(:last-child)::after {
    content: '/';
    margin-left: 0.5rem;
    color: #999;
}

.project-header {
    margin-bottom: 2rem;
}

.project-title-wrapper {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 1rem;
}

.project-status {
    padding: 0.25rem 0.75rem;
    border-radius: 1rem;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
}

.project-status.ongoing {
    background-color: #fff3cd;
    color: #856404;
}

.project-status.completed {
    background-color: #d4edda;
    color: #155724;
}

.project-meta {
    display: flex;
    flex-wrap: wrap;
    gap: 1.5rem;
    color: #555;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.meta-item i {
    color: var(--primary-color);
}

.project-main-image {
    margin: 2rem 0;
    border-radius: 0.5rem;
    overflow: hidden;
    box-shadow: 0 0.5rem 1rem rgba(0,0,0,0.1);
}

.project-main-image img {
    width: 100%;
    height: auto;
    display: block;
}

.project-description {
    margin: 3rem 0;
}

.project-description h2 {
    margin-bottom: 1.5rem;
    padding-bottom: 0.5rem;
    border-bottom: 2px solid var(--primary-color);
}

.related-projects {
    margin: 4rem 0;
}

.related-projects h2 {
    margin-bottom: 2rem;
    text-align: center;
}

.related-projects-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
    gap: 1.5rem;
}

.related-project {
    border-radius: 0.5rem;
    overflow: hidden;
    box-shadow: 0 0.25rem 0.5rem rgba(0,0,0,0.1);
    transition: transform 0.3s ease;
}

.related-project:hover {
    transform: translateY(-0.5rem);
}

.related-project img {
    width: 100%;
    height: 200px;
    object-fit: cover;
}

.related-project-info {
    padding: 1rem;
}

.related-project-info h3 {
    margin: 0 0 0.5rem;
    font-size: 1.1rem;
}

.related-client {
    margin: 0;
    color: #666;
    font-size: 0.9rem;
}

.project-cta {
    text-align: center;
    padding: 3rem;
    background-color: #f8f9fa;
    border-radius: 0.5rem;
    margin: 3rem 0;
}

.project-cta h3 {
    margin-bottom: 1rem;
}

.cta-buttons {
    display: flex;
    justify-content: center;
    gap: 1rem;
    margin-top: 1.5rem;
}

@media (max-width: 768px) {
    .project-meta {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .cta-buttons {
        flex-direction: column;
        align-items: center;
    }
}
</style>

<script>
// Simple image lightbox functionality
document.addEventListener('DOMContentLoaded', function() {
    const featuredImage = document.querySelector('.featured-image');
    if (featuredImage) {
        featuredImage.addEventListener('click', function() {
            const lightbox = document.createElement('div');
            lightbox.style.position = 'fixed';
            lightbox.style.top = '0';
            lightbox.style.left = '0';
            lightbox.style.width = '100%';
            lightbox.style.height = '100%';
            lightbox.style.backgroundColor = 'rgba(0,0,0,0.9)';
            lightbox.style.display = 'flex';
            lightbox.style.alignItems = 'center';
            lightbox.style.justifyContent = 'center';
            lightbox.style.zIndex = '1000';
            lightbox.style.cursor = 'zoom-out';
            
            const img = document.createElement('img');
            img.src = this.src;
            img.alt = this.alt;
            img.style.maxWidth = '90%';
            img.style.maxHeight = '90%';
            img.style.objectFit = 'contain';
            
            lightbox.appendChild(img);
            document.body.appendChild(lightbox);
            
            lightbox.addEventListener('click', function() {
                document.body.removeChild(lightbox);
            });
        });
    }
});
</script>